<?php
require_once '../../../config.php';

ini_set('display_errors', 1);
error_reporting(E_ALL);

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Start output buffering to capture any unintended output
ob_start();

// Set response headers to return a clean JSON response
header('Content-Type: application/json');
header('X-Content-Type-Options: nosniff');

// Retrieve POST data
$data = $_POST;

// Verify CSRF token for security
if (!isset($_SESSION['csrfToken'], $data['csrfToken']) || $data['csrfToken'] !== $_SESSION['csrfToken']) {
    ob_end_clean(); // Clear any buffered output
    echo json_encode([
        "success" => false,
        "message" => "CSRF token invalid",
        "debug" => [
            "session_csrf" => $_SESSION['csrfToken'] ?? 'NOT SET',
            "post_csrf" => $data['csrfToken'] ?? 'NOT SET'
        ]
    ]);
    exit;
}

// Validate required fields
if (!isset($data['user_id'], $data['username'], $data['email'])) {
    ob_end_clean();
    echo json_encode([
        "success" => false,
        "message" => "Missing required fields"
    ]);
    exit;
}

// Validate non-empty fields
if (empty($data['user_id']) || empty(trim($data['username'])) || empty(trim($data['email']))) {
    ob_end_clean();
    echo json_encode([
        "success" => false,
        "message" => "Empty required fields"
    ]);
    exit;
}

try {
    // Sanitize and prepare input values
    $user_id = (int)$data['user_id'];
    $username = trim($data['username']);
    $email = filter_var(trim($data['email']), FILTER_VALIDATE_EMAIL);

    // Prepare update data
    $updateData = [
        'username' => $username,
        'email' => $email
    ];

    // Optional password update
    if (!empty(trim($data['password']))) {
        $password = trim($data['password']);

        // Validate password strength
        if (strlen($password) < 8) {
            throw new Exception("Password must be at least 8 characters long");
        }

        $updateData['password'] = $password;
    }

    // Validate email
    if (!$email) {
        throw new Exception("Invalid email format");
    }

    // Validate username (e.g., alphanumeric, min/max length)
    if (!preg_match('/^[a-zA-Z0-9_]{3,20}$/', $username)) {
        throw new Exception("Invalid username format");
    }

    // Create an instance of UserManager using the database file path
    $db = PROJECT_DIR . '/data/users.db';
    $userManager = new UserManager($db);

    // Attempt to update the user details
    $result = $userManager->updateUser($user_id, $updateData);

    ob_end_clean(); // Clear any buffered output

    if ($result) {
        echo json_encode([
            "success" => true,
            "message" => "User updated successfully"
        ]);
    } else {
        echo json_encode([
            "success" => false,
            "message" => "User update failed"
        ]);
    }
} catch (Exception $e) {
    ob_end_clean(); // Clear any buffered output
    error_log("User update error: " . $e->getMessage());
    echo json_encode([
        "success" => false,
        "message" => $e->getMessage(), // More specific error message
        "error_details" => $e->getMessage() // Optional: for debugging
    ]);
}
exit;