<?php
class UserManager
{
    private ?PDO $pdo;

    /**
     * Constructor
     *
     * @param string $databasePath Path to the SQLite database file
     */
    public function __construct(string $databasePath)
    {
        SessionHelper::startSessionIfNotStarted();

        if (!file_exists($databasePath)) {
            die("DB not found");
        }
        $this->pdo = new PDO("sqlite:$databasePath");
        $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    }

    /**
     * Get all users from the database
     *
     * @return array All user data (excluding passwords)
     */
    public function getAllUsers(): array
    {
        try {
            $stmt = $this->pdo->prepare("SELECT id, username, email FROM users");
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Error getting all users: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Create a new user in the database
     *
     * @param array $data User data (username, password, email)
     * @return int User ID or 0 if creation fails
     */
    public function createUser(array $data): int
    {
        try {
            // Validate required fields
            $username = $data['username'] ?? null;
            $password = $data['password'] ?? null;
            $email = $data['email'] ?? null;

            if (!$username || !$password || !$email) {
                throw new Exception("Missing required user data");
            }

            // Check if username or email already exists
            $checkStmt = $this->pdo->prepare("SELECT id FROM users WHERE username = :username OR email = :email");
            $checkStmt->bindParam(':username', $username);
            $checkStmt->bindParam(':email', $email);
            $checkStmt->execute();
            if ($checkStmt->fetch()) {
                throw new Exception("Username or email already exists");
            }

            // Hash the password
            $hashedPassword = password_hash($password, PASSWORD_DEFAULT);

            // Insert new user
            $stmt = $this->pdo->prepare("INSERT INTO users (username, password, email) VALUES (:username, :password, :email)");
            $stmt->bindParam(':username', $username);
            $stmt->bindParam(':password', $hashedPassword);
            $stmt->bindParam(':email', $email);
            $stmt->execute();
            return $this->pdo->lastInsertId();
        } catch (PDOException $e) {
            error_log("Error creating user: " . $e->getMessage());
            return 0;
        } catch (Exception $e) {
            error_log("Error creating user: " . $e->getMessage());
            return 0;
        }
    }

    /**
     * Update a user in the database
     *
     * @param int $id User ID
     * @param array $data User data to update
     * @return bool True if the update was successful, false otherwise
     */
    public function updateUser(int $id, array $data): bool
    {
        try {
            // Prepare update query
            $updateFields = [];
            $params = [':id' => $id];

            // Check and prepare username update
            if (isset($data['username'])) {
                $updateFields[] = "username = :username";
                $params[':username'] = $data['username'];
            }

            // Check and prepare email update
            if (isset($data['email'])) {
                $updateFields[] = "email = :email";
                $params[':email'] = $data['email'];
            }

            // Check and prepare password update
            if (isset($data['password']) && !empty($data['password'])) {
                $updateFields[] = "password = :password";
                $params[':password'] = password_hash($data['password'], PASSWORD_DEFAULT);
            }

            // If no fields to update, return false
            if (empty($updateFields)) {
                return false;
            }

            // Construct and execute update query
            $query = "UPDATE users SET " . implode(', ', $updateFields) . " WHERE id = :id";
            $stmt = $this->pdo->prepare($query);
            $stmt->execute($params);

            return true;
        } catch (PDOException $e) {
            error_log("Error updating user: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Delete a user from the database
     *
     * @param int $id User ID
     * @return bool True if the deletion was successful, false otherwise
     */
    public function deleteUser(int $id): bool
    {
        try {
            $stmt = $this->pdo->prepare("DELETE FROM users WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            return true;
        } catch (PDOException $e) {
            error_log("Error deleting user: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Get a user from the database
     *
     * @param int $id User ID
     * @return array User data (excluding password)
     */
    public function getUser(int $id): array
    {
        try {
            $stmt = $this->pdo->prepare("SELECT id, username, email FROM users WHERE id = :id");
            $stmt->bindParam(':id', $id);
            $stmt->execute();
            return $stmt->fetch(PDO::FETCH_ASSOC) ?: [];
        } catch (PDOException $e) {
            error_log("Error getting user: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Authenticate a user
     *
     * @param string $username Username
     * @param string $password Password
     * @return array|null User data if authentication successful, null otherwise
     */
    public function authenticateUser(string $username, string $password): ?array
    {
        try {

            /*The filter_input function is not necessary in this case,
            as the input data is already validated and sanitized by the PDO prepared
            statement. The bindParam method will automatically escape any special
            characters in the input string, preventing SQL injection attacks.
            */

            $stmt = $this->pdo->prepare("SELECT * FROM users WHERE username = :username");
            $stmt->bindParam(':username', $username);
            $stmt->execute();
            $user = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($user && password_verify($password, $user['password'])) {
                // Remove password from returned data
                unset($user['password']);
                return $user;
            }
            return null;
        } catch (PDOException $e) {
            error_log("Error authenticating user: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Close the database connection
     */
    public function __destruct()
    {
        $this->pdo = null;
    }
}